/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_TROPHY_INTERFACE_H
#define _SCE_TROPHY_INTERFACE_H

#include "np_toolkit_defines.h"

namespace sce {
	namespace Toolkit{
		namespace NP{

			///	@brief
			///	The namespace containing trophy functionality.
			///
			///	The namespace containing trophy functionality.
			namespace Trophy {

			///	@brief
			///	Trophy interface class.
			///
			///	This class contains the set of static methods for managing trophies.
			class Interface
			{
			public:

				///	@brief
				///	Registers a trophy set.
				///
				///	Registers a trophy set. Because the registration process can take a while, the %NP %Toolkit library
				///	spawns a sub-thread to register the trophy set. The function provides the option to cache the trophy list or the trophy icons during the
				///	registration process.
				///
				/// This function can perform HDD size checking if the 
				///	required save data size is passed to <c><i>saveDataInBytes</i></c> and the user's PS3 HDD size is passed to <c><i>hddSizeInBytes</i></c>.
				/// The HDD size will be checked before the trophy set is installed, and an appropriate error message will be
				///	returned if there is a lack of HDD space.
				///	There is also an option to cache the trophy list at this point by setting
				///	<c><i>cacheTrophyList</i></c> to true.
				///
				///	@param cacheTrophyList			A flag that specifies whether the trophy list is to be cached or not.	
				///	@param cacheIcons				A flag that specifies whether the trophy icons are to be cached or not.
				///	@param saveDataInBytes			The required save data size. Optional. Pass in 0 if HDD size checking is not required.
				///	@param hddSizeInBytes			The user's PS3 HDD size. Optional. Pass in 0 if HDD size checking is not required.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS			The operation was successful.
				///	@retval TROPHY_NOT_ENOUGH_SPACE			The operation failed because there is not enough space on the HDD to install the trophy set.
				///
				///
				///	@par Example:
				///	@code
				///	int ret = sce::Toolkit::NP::Trophy::Interface::trophyRegisterSet(true, true, 0, 0);
				///	if (ret < 0) {
				///		printf("Trophy::Interface::trophyRegisterSet() failed, ret = 0x%x\n", ret);
				///	}
				///	@endcode
				static int trophyRegisterSet(bool cacheTrophyList, bool cacheIcons, int saveDataInBytes, int hddSizeInBytes);

				///	@brief
				///	Unlocks a particular trophy.
				///
				///	Unlocks a particular trophy. This function should be executed straight after a user has carried out an action
				///	that met the unlock criteria. A platinum trophy will also be automatically unlocked 
				///	if the last trophy in the trophy set was unlocked by this operation.
				///
				///	@param trophyId		The ID of the trophy that needs to be unlocked.
				///	@param async		A flag that indicates whether the function is non-blocking or blocking.
				///						Defaults to true so the function is non-blocking by default.
				///
				/// @retval	SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
				///	@retval SCE_TOOLKIT_NP_TROPHY_NOT_REGISTERED	The operation failed because the trophy set has not been registered yet.
				///	@retval	SCE_TOOLKIT_NP_TROPHY_BUSY				The operation failed because the trophy service is currently busy processing other requests.
				/// @retval	trophyUnlockSuccess					In the case of an asynchronous operation, this event will be passed to an event callback when the trophy has been unlocked successfully.
				/// @retval	trophyPlatinumUnlocked				In the case of an asynchronous operation, this event will be passed to an event callback when a platinum trophy has been unlocked.
				///	@retval	trophyUnlockFail					In the case of an asynchronous operation, this event will be passed to an event callback when a trophy has failed to unlock.
				static int trophyUnlock(int trophyId, bool async = true);

				///	@brief
				///	Retrieves a trophy list.
				///
				///	Retrieves a trophy list. This is filled with trophy information on all the
				///	trophies in the trophy set.
				///
				///	@param trophyList	A <c>Future</c> object, which receives the trophy information.
				///	@param async		A flag that indicates whether the function is non-blocking or blocking.
				///						Defaults to true so the function is non-blocking by default.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
				///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER			The operation failed because the <c>trophList</c> pointer was invalid.
				///	@retval SCE_TOOLKIT_NP_TROPHY_NOT_REGISTERED	The operation failed because the trophy set has not been registered yet.
				///	@retval	SCE_TOOLKIT_NP_TROPHY_BUSY				The operation failed because the trophy service is currently busy processing other requests.
				///	@retval SCE_TOOLKIT_NP_INIT_START_THREAD		The operation failed because the trophy service failed to spawn a thread for trophy information retrieval.
				///	@retval trophyListRetrievalSuccess			In the case of an asynchronous operation, this event will be passed to an event callback when the trophy list has been retrieved successfully.
				///	@retval trophyListRetrievalFail				In the case of an asynchronous operation, this event will be passed to an event callback when the trophy service has failed to retrieve the trophy list.
				///
				///	@par Example
				///	@code
				///	Future<TrophyGameInfo> gameInfo; 
				///	memset(gameInfo, 0x00, sizeof(TrophyGameInfo));
				///
				///	ret = Trophy::Interface::trophyRegister(saveDataSize, hddSize, cacheTrophyList, gameInfo);	// register and retrieve game trophy information
				///
				///	while(ret == SCE_TOOLKIT_NP_SUCCESS && !gameInfo->hasResult())
				///	{
				///		sys_timer_usleep(1000);
				///	}
				///
				///	int numTrophies = TrophyTest::gameInfo->get()->gameDetail.numTrophies;
				///	Future<TrophyInfo> * trophyList = new Future<TrophyInfo>[numTrophies];
				///	// Create numTrophies of Future<trophyInfo>
				///
				///	ret = Trophy::Interface::trophyRetrieveList(trophyList, false);
				///	
				///	int x = 0;
				///	for(int i = 0; i < numTrophies; i++) // Loop numTrophies of times to check
				///	{									// if their result is set before reading
				///		while(!trophyList[i].hasResult())// into them.
				///		{
				///			if(x++ % 1000 == 0){
				///				std::cout<<".";
				///				std::cout.flush();
				///			}
				///			sys_timer_usleep(1000);
				///		}
				///	}
				/// @endcode
				static int trophyRetrieveList(sce::Toolkit::NP::Utilities::Future<TrophyInfo> * trophyList, bool async = true);

				///	@brief
				///	Retrieves information on a game's trophy set.
				///
				///	Retrieves information on a game's trophy set. This includes the title's name,
				///	a description, the trophy set icon, the number of trophies available etc.
				///	
				///	@param gameInfo		A <c>Future</c> object, which receives the game information.
				///	@param async		A flag that indicates whether the function is non-blocking or blocking.
				///						Defaults to true so the function is non-blocking by default.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
				///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER			The operation failed because the <c>gameInfo</c> pointer was invalid.
				///	@retval SCE_TOOLKIT_NP_TROPHY_NOT_REGISTERED	The operation failed because the trophy set has not been registered yet.
				///	@retval	SCE_TOOLKIT_NP_TROPHY_BUSY				The operation failed because the trophy service is currently busy processing other requests.
				///	@retval SCE_TOOLKIT_NP_INIT_START_THREAD		The operation failed because the trophy service failed to spawn a thread for trophy information retrieval.
				/// @retval trophyGameInfoRetrievalSuccess		In the case of an asynchronous operation, this event will be passed to an event callback when the game's trophy set information has been retrieved successfully.
				///	@retval trophyGameInfoRetrievalFail			In the case of an asynchronous operation, this event will be passed to an event callback when the trophy service has failed to retrieve the game's trophy set information.
				///
				///	@par Example:
				///	@code
				///	// Assume that trophy set is registered already
				///
				///	Future<TrophyGameInfo> gameInfo;
				///	ret = Trophy::Interface::trophyRetrieveGame(&gameInfo, false);
				///	if(ret < 0){
				///		// Error handling here
				///	}
				/// @endcode
				static int trophyRetrieveGame(sce::Toolkit::NP::Utilities::Future<TrophyGameInfo> * gameInfo, bool async = true);

				///	@brief
				///	Retrieves the user's trophy progress.
				///
				///	Retrieves the user's trophy progress.
				///	
				///	@param trophyUserProgress		A <c>Future</c> object, which receives the user's trophy progress.
				///	@param async					A flag that indicates whether the function is non-blocking or blocking.
				///									Defaults to true so the function is non-blocking by default.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
				///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER			The operation failed because the <c>trophyUserProgress</c> pointer was invalid.
				///	@retval SCE_TOOLKIT_NP_TROPHY_NOT_REGISTERED	The operation failed because the trophy set has not been registered yet.
				///	@retval	SCE_TOOLKIT_NP_TROPHY_BUSY				The operation failed because the trophy service is currently busy processing other requests.
				/// @retval trophyProgressSuccess				In the case of an asynchronous operation, this event will be passed to an event callback when the user's trophy progress has been retrieved successfully.
				///	@retval trophyProgressFail					In the case of an asynchronous operation, this event will be passed to an event callback when the trophy service has failed to retrieve the user's trophy progress.
				///
				///	@par Example:
				///	@code
				///	// Assume that trophy set is registered already
				///
				///	Future<SceNpTrophyGameData> trophyUserProgress;
				///	ret = Trophy::Interface::trophyRetrieveProgress(&trophyUserProgress, false);
				///	if(ret < 0){
				///		// Error handling here
				///	}
				/// @endcode
				static int trophyRetrieveProgress(sce::Toolkit::NP::Utilities::Future<SceNpTrophyGameData> *trophyUserProgress, bool async = true);

				///	@brief
				///	Terminates the trophy service.
				///
				///	Terminates the trophy service. Internally, this function deallocates the memory used to cache trophy icons, trophy group information
				///	game information. It will also unload the trophy modules and calls the following termination functions: <c>sceNpTrophyDestroyHandle()</c>,
				///	<c>sceNpTrophyDestroyContext()</c> and <c>sceNpTrophyTerm()</c>. For the related error messages, please refer to these functions descriptions in the <em>%Trophy
				/// Reference</em> document.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS			The operation was successful.
				static int trophyTerm();

				///	@brief
				///	Initializes the trophy service.
				///
				///	Initializes the trophy service. This function does not need to be called when the %NP %Toolkit library first sets up the service.
				///	It should only be called after the trophy service has been explicitly terminated by calling trophyTerm(). Internally,
				///	the function loads the trophy modules and calls the following functions: <c>sceNpTrophyCreateContext()</c> and <c>sceNpTrophyCreateHandle()</c>.
				/// For the related error messages, please refer to these functions descriptions in the <em>%Trophy
				/// Reference</em> document.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS			The operation was successful.
				static int trophyInit();
			};

			}//end Trophy
		}//end NP
	}//end Toolkit
}//end sce

#endif
